import os
import tarfile
from minio import Minio
from minio.error import S3Error
from loguru import logger
import configparser

class MinioUploader:
    def __init__(self, config_path):
        """
        初始化 MinIO 上传器

        :param config_path: 配置文件路径
        """
        self.config = configparser.ConfigParser()
        self.config.read(config_path)
        self.minio_client = Minio(
            self.config['minio']['endpoint'],
            access_key=self.config['minio']['access_key'],
            secret_key=self.config['minio']['secret_key']
        )
        self.remote_folder = self.config['local']['remote_folder']
        self.bucket = self.config['minio']['bucket']

    def upload_file(self, file_path, object_name):
        """
        上传文件到 MinIO

        :param file_path: 本地文件路径
        :param object_name: MinIO 中的对象名称
        """
        try:
            self.minio_client.fput_object(
                self.bucket,
                object_name,
                file_path
            )
            logger.info(f"Uploaded {file_path} to {object_name}")
        except S3Error as e:
            logger.error(f"Failed to upload {file_path}: {e}")
        except Exception as e:
            logger.error(f"An error occurred: {e}")

    def process_tar_files(self):
        """
        处理 tar.gz 文件并上传到 MinIO
        """
        for file in os.listdir(self.remote_folder):
            if file.endswith('.tar.gz'):
                file_path = os.path.join(self.remote_folder, file)
                with tarfile.open(file_path, "r:gz") as tar:
                    for member in tar.getmembers():
                        f = tar.extractfile(member)
                        if f is not None:
                            temp_path = os.path.join(self.remote_folder, member.name)
                            with open(temp_path, 'wb') as temp_file:
                                temp_file.write(f.read())
                            self.upload_file(temp_path, member.name)
                            os.remove(temp_path)
                os.remove(file_path)

if __name__ == "__main__":
    uploader = MinioUploader("../config/config.ini")
    uploader.process_tar_files()