import React from "react";
import { Root, createRoot } from "react-dom/client";
import styles from "./styles/welcome.index.css?inline";
import { WithKiroSubscriptionPlansApiContext } from "./context/subscription-plans-api-context";
import { SubscriptionPlansPage } from "./components/subscription-plans-page/subscription-plans-page";
import { ColorTheme } from './types/shared-types';
import { SubscriptionPlansApi } from './types/subscription-plans-types';

export class KiroSubscriptionPlans extends HTMLElement {
	private root: Root | null = null;

	private api: SubscriptionPlansApi | undefined;
	private themes: ColorTheme[] = [];
	private currentThemeId: string = "";

	// Define the observed attributes
	static get observedAttributes(): string[] {
		return ["themetype"];
	}

	constructor() {
		super();
		this.attachShadow({ mode: "open" });
		this.root = this.root
			? this.root
			: createRoot(this.shadowRoot as ShadowRoot);
		const styleEl = document.createElement("style");
		styleEl.textContent = styles;
		this.shadowRoot?.appendChild(styleEl);
	}

	// Handle attribute changes
	attributeChangedCallback(name: string) {
		if (name === "themetype") {
			this.render();
		}
	}

	public setApi(api: SubscriptionPlansApi): void {
		this.api = api;
		this.render();
	}

	public setThemes(themes: ColorTheme[], currentThemeId: string): void {
		this.themes = themes;
		this.currentThemeId = currentThemeId;
		this.render();
	}

	private render() {
		if (!this.api) {
			return null;
		}
		this.root?.render(
			<React.StrictMode>
				<WithKiroSubscriptionPlansApiContext api={this.api}>
					<SubscriptionPlansPage
						themes={this.themes}
						currentThemeId={this.currentThemeId}
					/>
				</WithKiroSubscriptionPlansApiContext>
			</React.StrictMode>
		);
	}

	connectedCallback() {
		this.render();
	}
}

// Register the custom element
customElements.define("kiro-subscription-plans", KiroSubscriptionPlans);
