import {
	createContext,
	PropsWithChildren,
	ReactElement,
	useContext,
	useEffect,
	useMemo,
	useState,
} from "react";
import { OnboardingStep, OnboardingStepState } from "../types/shared-types";
import { ApiContext } from "./api-context";

type OnboardingStateSummary = Record<
	OnboardingStep,
	Omit<OnboardingStepState, "stepId">
>;

interface OnboardingStateContextType {
	onboardingState: OnboardingStateSummary;
	runOnboardingStep: (
		stepId: OnboardingStep,
		// This ensures that the step remains in a loading state for a minimum
		// amount of time, avoiding ultra brief flashes
		minDuration: number,
		input?: any
	) => Promise<OnboardingStepState>;
}

export const OnboardingStateContext = createContext<OnboardingStateContextType>(
	{} as OnboardingStateContextType
);

const ONBOARDING_STATE_POLLING_INTERVAL_SECONDS = 0.5;

const DEFAULT_ONBOARDING_STATE: OnboardingStateSummary = {
	"import-settings": {
		status: "incomplete",
	},
	"install-cli": {
		status: "incomplete",
	},
	"update-theme": {
		status: "complete",
	},
	"install-extensions": {
		status: "incomplete",
	},
};

export function WithOnboardingStateContext({
	children,
}: PropsWithChildren): ReactElement {
	const [onboardingState, setOnboardingState] =
		useState<OnboardingStateSummary>(DEFAULT_ONBOARDING_STATE);

	const { getOnboardingStates, executeOnboardingStep, getOnboardingStepState } =
		useContext(ApiContext);

	const initDeferred = useMemo(() => {
		const deferred = {} as { promise: Promise<void>; resolve: () => void };
		deferred.promise = new Promise((resolve) => {
			deferred.resolve = resolve;
		});
		return deferred;
	}, []);

	useEffect(() => {
		(async function () {
			const states = await getOnboardingStates();
			const newStates = { ...onboardingState };
			states.forEach((stepState) => {
				newStates[stepState.stepId] = stepState;
			});
			setOnboardingState(newStates);
			initDeferred.resolve();
		})();
	}, []);

	/**
	 * This function polls for new status messages from the extension.
	 * TODO: Can we make this a push instead of a pull?
	 */
	function pollForStatusMessages(stepId: OnboardingStep): void {
		const timeout = setInterval(async () => {
			const newState = await getOnboardingStepState(stepId);
			if (newState.status !== "loading") {
				clearInterval(timeout);
				return;
			}

			setOnboardingState((oldState) => {
				return {
					...oldState,
					[stepId]: {
						...oldState[stepId],
						messages: newState.messages,
					},
				};
			});
		}, ONBOARDING_STATE_POLLING_INTERVAL_SECONDS * 1000);
	}

	async function runOnboardingStep(
		stepId: OnboardingStep,
		minDuration: number,
		input?: any
	) {
		setOnboardingState((oldState) => ({
			...oldState,
			[stepId]: { ...oldState[stepId], status: "loading" },
		}));
		await initDeferred.promise;
		const promise = executeOnboardingStep(stepId, input);
		pollForStatusMessages(stepId);
		const [newState] = await Promise.all([
			promise,
			new Promise((resolve) => setTimeout(resolve, minDuration || 0)),
		]);
		setOnboardingState((oldState) => ({ ...oldState, [stepId]: newState }));
		return newState;
	}

	return (
		<OnboardingStateContext.Provider
			value={{ onboardingState, runOnboardingStep }}
		>
			{children}
		</OnboardingStateContext.Provider>
	);
}
