import { ReactElement, useContext } from "react";
import { KiroMainLogoTitle } from "../kiro-main-logo-title";
import { KiroButton } from "../kiro-button";
import {
	RecentEntry,
	WelcomeAction,
	WelcomeActions,
} from "../../types/shared-types";
import { LinkButton } from "../link-button";
import { WelcomeApiContext } from "../../context/welcome-api-context";

interface WelcomePageProps {
	recentEntries: RecentEntry[];
	actions: WelcomeActions;
}

interface ActionMetadata {
	label: string;
	icon: ReactElement;
}

const actionMap: Record<WelcomeAction["id"], ActionMetadata> = {
	topLevelGitClone: {
		label: "Clone repository",
		icon: <i className="codicon codicon-source-control"></i>,
	},
	topLevelRemoteOpen: {
		label: "Connect to",
		icon: <i className="codicon codicon-remote"></i>,
	},
};

export function WelcomePage({
	recentEntries,
	actions,
}: WelcomePageProps): ReactElement {
	const { openRecentEntry, executeCommand } = useContext(WelcomeApiContext);

	function handleRecentEntryClick(
		event: React.MouseEvent<HTMLButtonElement, MouseEvent>,
		entry: RecentEntry
	) {
		openRecentEntry(entry, event.ctrlKey || event.metaKey);
	}

	return (
		<div className="vscode-dark w-full h-full flex items-center justify-center text-[var(--vscode-foreground)] bg-[var(--vscode-welcomePage-background)] overflow-hidden">
			<div className="p-10 box-border basis-[480px] shrink">
				<div className="mb-5">
					<KiroMainLogoTitle description="An agentic IDE that helps you do your best work." />
				</div>
				<div className="flex flex-col gap-2 @container">
					<h2 className="text-[16px]">Get started</h2>
					<KiroButton
						disabled={!actions.mainAction}
						onClick={() =>
							actions.mainAction && executeCommand(actions.mainAction.command)
						}
					>
						Open a project
					</KiroButton>
					<div className="bg-[var(--vscode-welcomePage-tileBackground)] p-3 rounded-lg">
						<h3 className="text-[12px] font-normal">Recent projects</h3>
						<ul className="list-none text-[12px]">
							{recentEntries.map((entry) => {
								return (
									<li
										className="flex flex-row gap-2"
										key={`${entry.parentPath}/${entry.name}`}
									>
										<div className="w-[50%]">
											<LinkButton
												onClick={(event) =>
													handleRecentEntryClick(event, entry)
												}
											>
												<div className="text-left">{entry.name}</div>
											</LinkButton>
										</div>

										<div className="text-right w-[50%] whitespace-nowrap text-ellipsis overflow-hidden">
											{entry.parentPath}
										</div>
									</li>
								);
							})}
						</ul>
					</div>
					<ul className="flex flex-col gap-2 justify-start list-none @sm:flex-row">
						{actions.secondaryActions.map((action) => {
							const mappedData = actionMap[action.id];
							if (!mappedData) {
								return null;
							}
							return (
								<li className="grow" key={action.id}>
									<KiroButton
										style="secondary"
										onClick={() => executeCommand(action.command)}
									>
										<div className="flex flex-row items-center justify-center gap-2">
											{mappedData.icon}
											<span className="text-[12px]">{mappedData.label}</span>
										</div>
									</KiroButton>
								</li>
							);
						})}
					</ul>
				</div>
			</div>
		</div>
	);
}
