import { ReactElement, memo } from "react";
import { SubscriptionType } from "../../types/account-dashboard-types";
import { UISubscriptionPlan } from "../../utils/subscription-plan-transformer";

interface PlanCardProps {
	plan: UISubscriptionPlan;
	onUpgrade: (subscriptionType: SubscriptionType) => void;
}

export const PlanCard = memo(function PlanCard({
	plan,
	onUpgrade,
}: PlanCardProps): ReactElement {
	const handleUpgrade = () => {
		if (plan.subscriptionType) {
			onUpgrade(plan.subscriptionType);
		}
	};

	return (
		<div
			className="relative rounded-lg p-6 flex flex-col"
			style={{
				borderRadius: "var(--Corner-radius-radius-3, 16px)",
				border: "1px solid var(--Prey-600, #4A464F)",
			}}
		>
			{/* Plan Header */}
			<div className="mb-6">
				<div className="flex items-center justify-between mb-4">
					<h3 className="text-[var(--vscode-foreground)] font-medium text-base">
						{plan.name}
					</h3>
					{/* Current Plan Badge */}
					{plan.isCurrent && (
						<div className="bg-green-800 text-white px-3 py-1 rounded text-xs font-medium">
							Current plan
						</div>
					)}
				</div>

				{/* Gray divider */}
				<div className="border-t border-solid border-[var(--Prey-600, #4A464F)] mb-4"></div>
				<div className="mb-4">
					<div className="text-[var(--vscode-foreground)] text-2xl font-bold mb-1">
						{plan.price}
					</div>
					<div className="text-[var(--vscode-descriptionForeground)] text-sm">
						{plan.subtitle}
					</div>
				</div>
			</div>

			{/* Description */}
			{plan.description && (
				<p className="text-[var(--vscode-descriptionForeground)] text-sm mb-4">
					{plan.description}
				</p>
			)}

			{/* Features */}
			<div className="space-y-2 mb-6 flex-1">
				{plan.features.map((feature, featureIndex) => (
					<div key={featureIndex} className="flex items-start gap-2">
						<i
							className="codicon codicon-check text-[var(--color-kiro-purple-primary)] mt-0.5 font-bold"
							style={{ fontSize: "16px" }}
						></i>

						<span className="text-[var(--vscode-foreground)] text-sm leading-relaxed">
							{feature}
						</span>
					</div>
				))}
			</div>

			{/* Action Button */}
			{plan.buttonText && plan.subscriptionType !== 'Q_DEVELOPER_STANDALONE' && !plan.isCurrent && (
				<div className="mt-auto">
					<button
						onClick={handleUpgrade}
						className="w-full py-2 px-4 rounded-sm text-sm font-medium transition-colors text-[var(--vscode-foreground)] hover:bg-[var(--color-kiro-purple-primary)] hover:text-white"
						style={{
							border: "1px solid var(--Prey-600, #4A464F)",
						}}
					>
						{plan.buttonText}
					</button>
				</div>
			)}
		</div>
	);
});
