import { describe, it, expect, vi, beforeEach } from "vitest";
import { render, screen, fireEvent } from "@testing-library/react";
import { SubscriptionPlansPage } from "../subscription-plans-page";
import { WithKiroSubscriptionPlansApiContext } from "../../../context/subscription-plans-api-context";
import { useSubscriptionPlans } from "../../../hooks/useSubscriptionPlans";
import { mockSubscriptionPlans } from "./mockSubscriptionPlansData";

// Use mock data from the separate file
const testSubscriptionPlans =
	mockSubscriptionPlans.happyPathCase.data!.subscriptionPlans;
const testDisclaimers = mockSubscriptionPlans.happyPathCase.data!.disclaimers;
import type { SubscriptionPlansApi } from "../../../types/subscription-plans-types";
import type { ColorTheme } from "../../../types/shared-types";

// Mock the useSubscriptionPlans hook
vi.mock("../../../hooks/useSubscriptionPlans");

const mockUseSubscriptionPlans = vi.mocked(useSubscriptionPlans);

// Mock API context
const mockApi: SubscriptionPlansApi = {
	executeCommand: vi.fn(),
	getSubscriptionPlans: vi.fn(),
	getCheckoutSessionUrl: vi.fn(),
	close: vi.fn(),
	showDashboard: vi.fn(),
	openExternalLink: vi.fn(),
};

// Mock themes
const mockThemes: ColorTheme[] = [
	{
		id: "dark",
		type: "dark",
		label: "",
		getColor: function (colorName: string): string {
			throw new Error("Function not implemented.");
		},
		getTokenCSS: function (parentClass: string): string {
			throw new Error("Function not implemented.");
		},
	},
	{
		id: "light",
		type: "light",
		label: "",
		getColor: function (colorName: string): string {
			throw new Error("Function not implemented.");
		},
		getTokenCSS: function (parentClass: string): string {
			throw new Error("Function not implemented.");
		},
	},
];

// Helper function to render component with context
const renderWithContext = (api = mockApi, currentThemeId = "dark") => {
	return render(
		<WithKiroSubscriptionPlansApiContext api={api}>
			<SubscriptionPlansPage
				themes={mockThemes}
				currentThemeId={currentThemeId}
			/>
		</WithKiroSubscriptionPlansApiContext>
	);
};

describe("SubscriptionPlansPage", () => {
	beforeEach(() => {
		vi.clearAllMocks();
	});

	describe("Loading State", () => {
		it("should show loading skeleton when loading", () => {
			mockUseSubscriptionPlans.mockReturnValue({
				subscriptionPlans: [],
				disclaimers: [],
				loading: true,
				error: null,
				handleUpgrade: vi.fn(),
				clearError: vi.fn(),
			});

			renderWithContext();

			expect(
				screen.getByText("Loading subscription plans...")
			).toBeInTheDocument();
			// Check for loading spinner icon
			expect(document.querySelector(".codicon-loading")).toBeInTheDocument();
		});
	});

	describe("Error States", () => {
		it("should show error banner when there is an error", () => {
			const mockClearError = vi.fn();
			mockUseSubscriptionPlans.mockReturnValue({
				subscriptionPlans: [],
				disclaimers: [],
				loading: false,
				error: "Failed to load subscription plans",
				handleUpgrade: vi.fn(),
				clearError: mockClearError,
			});

			renderWithContext();

			expect(
				screen.getByText("Failed to load subscription plans")
			).toBeInTheDocument();

			// Test dismiss functionality
			const dismissButton = screen.getByRole("button", {
				name: /dismiss error/i,
			});
			fireEvent.click(dismissButton);
			expect(mockClearError).toHaveBeenCalledTimes(1);
		});

		it("should show no plans available message when there are no plans", () => {
			const mockClearError = vi.fn();
			mockUseSubscriptionPlans.mockReturnValue({
				subscriptionPlans: [],
				disclaimers: [],
				loading: false,
				error: null,
				handleUpgrade: vi.fn(),
				clearError: mockClearError,
			});

			renderWithContext();

			expect(
				screen.getByText("No Subscription plan available")
			).toBeInTheDocument();
		});
	});

	describe("Successful Plans Display", () => {
		it("should display subscription plans correctly", () => {
			mockUseSubscriptionPlans.mockReturnValue({
				subscriptionPlans: testSubscriptionPlans,
				disclaimers: testDisclaimers,
				loading: false,
				error: null,
				handleUpgrade: vi.fn(),
				clearError: vi.fn(),
			});

			renderWithContext();

			// Check plan titles
			expect(screen.getByText("KIRO FREE")).toBeInTheDocument();
			expect(screen.getByText("KIRO PRO")).toBeInTheDocument();
			expect(screen.getByText("KIRO PRO+")).toBeInTheDocument();
			expect(screen.getByText("KIRO POWER")).toBeInTheDocument();

			// Check pricing
			expect(screen.getByText("$0")).toBeInTheDocument();
			expect(screen.getByText("$20")).toBeInTheDocument();
			expect(screen.getByText("$40")).toBeInTheDocument();
			expect(screen.getByText("$200")).toBeInTheDocument();

			// Check features
			expect(screen.getByText("50 vibe requests")).toBeInTheDocument();
			expect(screen.getByText("Up to 225 vibe requests")).toBeInTheDocument();
			expect(screen.getByText("Up to 450 vibe requests")).toBeInTheDocument();
			expect(screen.getByText("Up to 2250 vibe requests")).toBeInTheDocument();
		});

		it("should show current plan badge for free plan", () => {
			mockUseSubscriptionPlans.mockReturnValue({
				subscriptionPlans: testSubscriptionPlans,
				disclaimers: testDisclaimers,
				loading: false,
				error: null,
				handleUpgrade: vi.fn(),
				clearError: vi.fn(),
			});

			renderWithContext();

			expect(screen.getByText("Current plan")).toBeInTheDocument();
		});

		it("should show upgrade buttons for non-current plans", () => {
			mockUseSubscriptionPlans.mockReturnValue({
				subscriptionPlans: testSubscriptionPlans,
				disclaimers: testDisclaimers,
				loading: false,
				error: null,
				handleUpgrade: vi.fn(),
				clearError: vi.fn(),
			});

			renderWithContext();

			expect(screen.getByText("Upgrade to Pro")).toBeInTheDocument();
			expect(screen.getByText("Upgrade to Pro+")).toBeInTheDocument();
			expect(screen.getByText("Upgrade to Power")).toBeInTheDocument();
			// Free plan should not have upgrade button since it's current
			expect(screen.queryByText("Upgrade to Free")).not.toBeInTheDocument();
		});

		it("should call handleUpgrade when upgrade button is clicked", () => {
			const mockHandleUpgrade = vi.fn();
			mockUseSubscriptionPlans.mockReturnValue({
				subscriptionPlans: testSubscriptionPlans,
				disclaimers: testDisclaimers,
				loading: false,
				error: null,
				handleUpgrade: mockHandleUpgrade,
				clearError: vi.fn(),
			});

			renderWithContext();

			const upgradeButton = screen.getByText("Upgrade to Pro");
			fireEvent.click(upgradeButton);

			expect(mockHandleUpgrade).toHaveBeenCalledWith(
				"Q_DEVELOPER_STANDALONE_PRO"
			);
		});

		it("should display disclaimers", () => {
			mockUseSubscriptionPlans.mockReturnValue({
				subscriptionPlans: testSubscriptionPlans,
				disclaimers: testDisclaimers,
				loading: false,
				error: null,
				handleUpgrade: vi.fn(),
				clearError: vi.fn(),
			});

			renderWithContext();

			expect(
				screen.getByText(
					"Need a little bit more? Paid plans can turn on overage billing Prices are exclusive of applicable taxes and duties, including VAT and applicable sales tax"
				)
			).toBeInTheDocument();
		});

		it("should display currency symbols correctly", () => {
			mockUseSubscriptionPlans.mockReturnValue({
				subscriptionPlans: testSubscriptionPlans,
				disclaimers: testDisclaimers,
				loading: false,
				error: null,
				handleUpgrade: vi.fn(),
				clearError: vi.fn(),
			});

			renderWithContext();

			// Check that USD currency symbols are displayed correctly
			expect(screen.getByText("$0")).toBeInTheDocument();
			expect(screen.getByText("$20")).toBeInTheDocument();
			expect(screen.getByText("$40")).toBeInTheDocument();
			expect(screen.getByText("$200")).toBeInTheDocument();
		});

		it("should display page title", () => {
			mockUseSubscriptionPlans.mockReturnValue({
				subscriptionPlans: testSubscriptionPlans,
				disclaimers: testDisclaimers,
				loading: false,
				error: null,
				handleUpgrade: vi.fn(),
				clearError: vi.fn(),
			});

			renderWithContext();

			expect(screen.getByText("Plan options")).toBeInTheDocument();
		});
	});

	describe("Theme Support", () => {
		it("should apply dark theme class by default", () => {
			mockUseSubscriptionPlans.mockReturnValue({
				subscriptionPlans: testSubscriptionPlans,
				disclaimers: testDisclaimers,
				loading: false,
				error: null,
				handleUpgrade: vi.fn(),
				clearError: vi.fn(),
			});

			const { container } = renderWithContext(mockApi, "dark");

			expect(container.querySelector(".vscode-dark")).toBeInTheDocument();
		});

		it("should apply light theme class when specified", () => {
			mockUseSubscriptionPlans.mockReturnValue({
				subscriptionPlans: testSubscriptionPlans,
				disclaimers: testDisclaimers,
				loading: false,
				error: null,
				handleUpgrade: vi.fn(),
				clearError: vi.fn(),
			});

			const { container } = renderWithContext(mockApi, "light");

			expect(container.querySelector(".vscode-light")).toBeInTheDocument();
		});

		it("should fallback to dark theme for unknown theme id", () => {
			mockUseSubscriptionPlans.mockReturnValue({
				subscriptionPlans: testSubscriptionPlans,
				disclaimers: testDisclaimers,
				loading: false,
				error: null,
				handleUpgrade: vi.fn(),
				clearError: vi.fn(),
			});

			const { container } = renderWithContext(mockApi, "unknown-theme");

			expect(container.querySelector(".vscode-dark")).toBeInTheDocument();
		});
	});

	describe("Edge Cases", () => {
		it("should handle plans without features", () => {
			const plansWithoutFeatures = [
				{
					...testSubscriptionPlans[0],
					description: {
						...testSubscriptionPlans[0].description,
						features: [],
					},
				},
			];

			mockUseSubscriptionPlans.mockReturnValue({
				subscriptionPlans: plansWithoutFeatures,
				disclaimers: testDisclaimers,
				loading: false,
				error: null,
				handleUpgrade: vi.fn(),
				clearError: vi.fn(),
			});

			renderWithContext();

			expect(screen.getByText("KIRO FREE")).toBeInTheDocument();
			// Should not crash when no features are present
		});

		it("should not show upgrade button for standalone subscription type", () => {
			const standaloneSubscriptionPlans = [
				{
					...testSubscriptionPlans[1],
					subscriptionType: "Q_DEVELOPER_STANDALONE" as const,
				},
			];

			mockUseSubscriptionPlans.mockReturnValue({
				subscriptionPlans: standaloneSubscriptionPlans,
				disclaimers: testDisclaimers,
				loading: false,
				error: null,
				handleUpgrade: vi.fn(),
				clearError: vi.fn(),
			});

			renderWithContext();

			expect(screen.queryByText("Upgrade to Pro")).not.toBeInTheDocument();
		});
	});
});
