import { ReactElement, useContext, useState } from "react";
import { KiroWordmarkLogo } from "../kiro-logo";
import { SignInSteps } from "../sign-in-steps";
import { SignInStatus, SignInOptions } from "../../types/sign-in-types";
import { ApiContext } from "../../context";
import { ThemeType } from "../../types/shared-types";
import { SignInInvitation } from "../sign-in-invitation";

interface SignInProps {
	isInternalUser: boolean;
	themeType: ThemeType;
	onSignInComplete: () => void;
}

type SignInView = "default" | "invitation";

/**
 *  Main Sign-View
 */
export default function SignIn({
	isInternalUser,
	themeType,
	onSignInComplete,
}: SignInProps): ReactElement {
	const [authStatus, setAuthStatus] = useState<SignInStatus>("not-started");
	const [authError, setAuthError] = useState<string>("");
	const [signInView, setSignInView] = useState<SignInView>("default");
	const [signInOptions, setSignInOptions] = useState<
		SignInOptions | undefined
	>();

	const { signIn, cancelSignIn } = useContext(ApiContext);

	async function socialSignIn(signInOptions: SignInOptions): Promise<void> {
		if (authStatus === "pending" || authStatus === "success") {
			return;
		}
		setAuthStatus("pending");
		try {
			await signIn(signInOptions);
			onSignInComplete();
			// eslint-disable-next-line @typescript-eslint/no-unused-vars
		} catch (error) {
			// When user cancels the flow, we just go back to the beginning
			if (
				error &&
				typeof error === "object" &&
				"name" in error &&
				error.name === "CANCELED"
			) {
				setAuthStatus("not-started");
			} else if (
				error &&
				typeof error === "object" &&
				"name" in error &&
				error.name === "SIGN_IN_BLOCKED"
			) {
				setSignInOptions(signInOptions);
				setSignInView("invitation");
				setAuthStatus("not-started");
			} else {
				let errorMessage = "";
				if (
					error &&
					typeof error === "object" &&
					"name" in error &&
					error.name === "GENERIC_USER_FACING_KIRO_ERROR" &&
					"message" in error
				) {
					errorMessage = `${error.message}`;
				}
				setAuthError(errorMessage);
				setAuthStatus("error");
			}
		}
	}

	return (
		<div className="h-full w-full overflow-y-auto">
			<div
				className={`min-h-full w-full relative flex items-center justify-center p-10 box-border`}
			>
				<div className="relative w-full flex flex-col items-center gap-10 pt-20">
					{signInView === "invitation" && signInOptions ? (
						<SignInInvitation
							onSignIn={(options) => {
								void socialSignIn(options);
							}}
							themeType={themeType}
							errorText={authError}
							onCancel={() => {
								setAuthError("");
								setAuthStatus("not-started");
								setSignInView("default");
								cancelSignIn();
							}}
							signInOptions={signInOptions}
							signInStatus={authStatus}
						/>
					) : (
						<>
							<h1 className="w-[400px]">
								<KiroWordmarkLogo
									signInStatus={authStatus}
									themeType={themeType}
								/>
							</h1>
							<div className="text-left" style={{ width: "350px" }}>
								<SignInSteps
									signInStatus={authStatus}
									signInError={authError}
									isInternalUser={isInternalUser}
									themeType={themeType}
									onSignIn={(options) => {
										void socialSignIn(options);
									}}
									onCancel={() => {
										cancelSignIn();
									}}
								/>
							</div>
						</>
					)}
				</div>
			</div>
		</div>
	);
}
