import {
	render,
	screen,
	fireEvent,
	waitFor,
	RenderResult,
} from "@testing-library/react";
import { vi } from "vitest";
import SignIn from "./sign-in.view";

describe("SignIn", () => {
	const mockCallApi = vi.fn();

	beforeEach(() => {
		mockCallApi.mockReset();
	});

	const renderSignIn = (): RenderResult => {
		return render(
			<SignIn
				isInternalUser
				themeType="dark"
				onSignInComplete={() => Promise.resolve()}
			/>
		);
	};

	it("renders the sign in page with logo", () => {
		renderSignIn();
		expect(screen.getByTestId("kiro-logo")).toBeInTheDocument();
	});

	it("handles social sign in flow successfully", async () => {
		mockCallApi.mockResolvedValueOnce(undefined);
		renderSignIn();

		const googleButton = screen.getByRole("button", { name: /Google/i });
		fireEvent.click(googleButton);

		expect(mockCallApi).toHaveBeenCalledWith("authenticateWithOptions", {
			provider: "Google",
			authMethod: "social",
		});

		await waitFor(() => {
			expect(screen.getByText(/You are signed in/i)).toBeInTheDocument();
		});
	});

	it("handles social sign in error state", async () => {
		mockCallApi.mockRejectedValueOnce(new Error("Auth failed"));
		renderSignIn();

		const googleButton = screen.getByRole("button", { name: /Google/i });
		fireEvent.click(googleButton);

		await waitFor(() => {
			expect(screen.getByText(/error/i)).toBeInTheDocument();
		});
	});

	it("prevents multiple sign in attempts while pending", () => {
		mockCallApi.mockImplementation(() => new Promise(() => {})); // Never resolves
		renderSignIn();

		const googleButton = screen.getByRole("button", { name: /Google/i });
		fireEvent.click(googleButton);
		fireEvent.click(googleButton); // Second click should be ignored

		expect(mockCallApi).toHaveBeenCalledTimes(1);
	});
});
