import { SignInButton } from "../sign-in-button";
import { ReactElement, useState } from "react";
import { SignInStatus, SignInOptions } from "../../types/sign-in-types";
import { SignInInput } from "../sign-in-input";
import { KiroButton } from "../kiro-button";
import { LinkButton } from "../link-button";
import { ThemeType } from "../../types/shared-types";

interface EnterpriseSignInValues {
	region: string;
	url: string;
}

interface SignInStepsProps {
	signInStatus: SignInStatus;
	signInError: string;
	isInternalUser: boolean;
	themeType: ThemeType;
	onSignIn: (signInOptions: SignInOptions) => void;
	onCancel: () => void;
}

interface SocialLoginStepProps {
	signInStatus: SignInStatus;
	signInError: string;
	isInternalUser: boolean;
	themeType: ThemeType;
	onSignIn: (signInOptions: SignInOptions) => void;
	onCancel: () => void;
	onEnterpriseSelect: () => void;
}

interface EnterpriseStepProps {
	values: EnterpriseSignInValues;
	signInStatus: SignInStatus;
	signInError: string;
	onSubmit: (payload: EnterpriseSignInValues) => void;
	onBack: () => void;
	onChange: (values: EnterpriseSignInValues) => void;
	onCancel: () => void;
}

interface ErrorMessageProps {
	content: string;
}

interface PendingMessageProps {
	onCancel: () => void;
}

type SignInStep = "social-sign-in" | "enterprise";

function ErrorMessage({ content }: ErrorMessageProps): ReactElement {
	return (
		<p className="text-red-300 text-[14px]/7.5 font-medium text-wrap mb-5">
			{ content || 'There was an error signing you in. Please try again.' }
		</p>
	);
}

function SuccessMessage(): ReactElement {
	return (
		<p className="text-white text-[20px]/7.5 font-medium text-wrap">
			You are signed in!
		</p>
	);
}

function PendingMessage({ onCancel }: PendingMessageProps): ReactElement {
	return (
		<div className="gap-5 flex flex-col items-end">
			<p className="secondary-text text-[20px]/7.5 font-normal text-wrap mb-5">
				We will take you to your default web browser to sign in and bring you
				back when it's complete...
			</p>
			<div className="w-30">
				<KiroButton type="button" onClick={onCancel}>
					Cancel
				</KiroButton>
			</div>
		</div>
	);
}

function WelcomeMessage(): ReactElement {
	return (
		<p className="secondary-text text-[20px]/7.5 font-normal text-wrap mb-5 text-center">
			An agentic IDE that helps you do your best work.
		</p>
	);
}

function SocialLoginStep({
	onSignIn,
	onEnterpriseSelect,
	themeType,
	onCancel,
	signInStatus,
	signInError,
	isInternalUser,
}: SocialLoginStepProps): ReactElement {
	const AllowedSignInProviders: SignInOptions["provider"][] = [
		"BuilderId",
		"Enterprise",
		"Internal",
		"Google",
		"Github",
	];

	return (
		<>
			{signInStatus === "error" && <ErrorMessage content={signInError} />}
			{signInStatus === "success" && <SuccessMessage />}
			{signInStatus === "pending" && <PendingMessage onCancel={onCancel} />}
			{signInStatus === "not-started" && <WelcomeMessage />}
			{(signInStatus === "error" || signInStatus === "not-started") && (
				<div className="w-[350px] min-h-30 mt-5">
					<ul className="list-none">
						{AllowedSignInProviders.includes("Internal") && isInternalUser && (
							<li className="mb-4">
								<SignInButton
									provider="Internal"
									themeType={themeType}
									onClick={() => {
										onSignIn({ authMethod: "IdC", provider: "Internal" });
									}}
								/>
							</li>
						)}
						{AllowedSignInProviders.includes("Google") && (
							<li className="mb-4">
								<SignInButton
									provider="Google"
									themeType={themeType}
									onClick={() => {
										onSignIn({ authMethod: "social", provider: "Google" });
									}}
								/>
							</li>
						)}
						{AllowedSignInProviders.includes("Github") && (
							<li className="mb-4">
								<SignInButton
									provider="Github"
									themeType={themeType}
									onClick={() => {
										onSignIn({ authMethod: "social", provider: "Github" });
									}}
								/>
							</li>
						)}
						{AllowedSignInProviders.includes("BuilderId") && (
							<li className="mb-4">
								<SignInButton
									provider="BuilderId"
									themeType={themeType}
									onClick={() => {
										onSignIn({ authMethod: "IdC", provider: "BuilderId" });
									}}
								/>
							</li>
						)}
						{AllowedSignInProviders.includes("Enterprise") && (
							<li className="mb-4">
								<SignInButton
									provider="Enterprise"
									themeType={themeType}
									onClick={onEnterpriseSelect}
								/>
							</li>
						)}
					</ul>
				</div>
			)}
			<div className="text-center mt-5 text-[var(--vscode-disabledForeground)] text-xs">
				By signing in, you agree to the{" "}
				<a
					className="underline text-[var(--vscode-textLink-foreground)]"
					href="https://aws.amazon.com/agreement/"
					target="_blank"
				>
					AWS Customer Agreement
				</a>
				,{" "}
				<a
					className="underline text-[var(--vscode-textLink-foreground)]"
					href="https://aws.amazon.com/service-terms/"
					target="_blank"
				>
					Service Terms
				</a>
				, and{" "}
				<a
					className="underline text-[var(--vscode-textLink-foreground)]"
					href="https://aws.amazon.com/privacy/"
					target="_blank"
				>
					Privacy Notice
				</a>
				.
			</div>
		</>
	);
}

function EnterpriseStep({
	values,
	onChange,
	onSubmit,
	onBack,
	onCancel,
	signInStatus,
	signInError
}: EnterpriseStepProps): ReactElement {
	return (
		<>
			<button
				onClick={onBack}
				className={`cursor-pointer text-kiro-button-link mb-3 ${
					signInStatus === "pending" || signInStatus === "success"
						? "invisible"
						: ""
				}`}
			>
				{
					// Using !important here because codicon comes with a default font-size of 16px but its CSS definition is using
					// a higher specificity than what tailwind is usually producing.
				}
				<i className="codicon codicon-arrow-left !text-[24px]/10"></i>
			</button>
			<h1 className="text-[26px] font-normal mb-5">
				Sign in with AWS IAM Identity Center
			</h1>
			{signInStatus === "error" && <ErrorMessage content={signInError} />}
			{signInStatus === "success" && <SuccessMessage />}
			{signInStatus === "pending" && <PendingMessage onCancel={onCancel} />}
			{(signInStatus === "error" || signInStatus === "not-started") && (
				<form
					className="w-[320px] gap-5 flex flex-col"
					onSubmit={() => {
						onSubmit(values);
					}}
				>
					<SignInInput
						id="enterprise-sign-in-url"
						label="Start URL"
						name="url"
						helperText="Provided by an admin or help desk."
						placeholder="your_subdomain.awsapps.com/start"
						value={values.url}
						onChange={(event) => {
							onChange({ region: values.region, url: event.target.value });
						}}
						autoFocus
					/>
					<SignInInput
						id="enterprise-sign-in-region"
						label="Region"
						name="region"
						helperText="AWS Region that hosts Identity directory."
						value={values.region}
						onChange={(event) => {
							onChange({ region: event.target.value, url: values.url });
						}}
					/>
					<KiroButton disabled={!values.url} type="submit">
						Continue
					</KiroButton>
					<p className="font-normal text-[12px]/4">
						Not an enterprise customer?{" "}
						<LinkButton onClick={onBack}>View all login options</LinkButton>
					</p>
				</form>
			)}
		</>
	);
}

/**
 * Container for all sign-in buttons
 */
export function SignInSteps({
	signInStatus,
	signInError,
	isInternalUser,
	themeType,
	onSignIn,
	onCancel,
}: SignInStepsProps): ReactElement {
	const [step, setStep] = useState<SignInStep>("social-sign-in");

	const [enterpriseFormValues, setEnterpriseFormValues] =
		useState<EnterpriseSignInValues>({
			url: "",
			region: "us-east-1",
		});

	if (step === "social-sign-in") {
		return (
			<SocialLoginStep
				signInStatus={signInStatus}
				signInError={signInError}
				isInternalUser={isInternalUser}
				themeType={themeType}
				onSignIn={onSignIn}
				onCancel={onCancel}
				onEnterpriseSelect={() => {
					setStep("enterprise");
				}}
			/>
		);
	} else {
		return (
			<EnterpriseStep
				signInStatus={signInStatus}
				signInError={signInError}
				values={enterpriseFormValues}
				onChange={setEnterpriseFormValues}
				onBack={() => {
					setStep("social-sign-in");
				}}
				onSubmit={(values) => {
					onSignIn({
						authMethod: "IdC",
						provider: "Enterprise",
						startUrl: values.url.startsWith("https://")
							? values.url
							: `https://${values.url}`,
						region: values.region || "us-east-1",
					});
				}}
				onCancel={onCancel}
			/>
		);
	}
}
