import { render, screen, fireEvent } from "@testing-library/react";
import { SignInSteps } from "./sign-in-steps";
import { vi } from "vitest";

describe("SignInSteps", () => {
	const mockOnSocialSignIn = vi.fn();
	const mockOnEnterpriseSignIn = vi.fn();
	const mockOnCancel = vi.fn();

	beforeEach(() => {
		vi.clearAllMocks();
	});

	it("renders social login step by default", () => {
		render(
			<SignInSteps
				signInStatus="not-started"
				signInError=""
				themeType="dark"
				onSignIn={mockOnSocialSignIn}
				onCancel={mockOnCancel}
				isInternalUser
			/>
		);

		expect(screen.getByText(/Pro license\?/)).toBeInTheDocument();
	});

	it("shows pending message during sign in", () => {
		render(
			<SignInSteps
				signInStatus="pending"
				signInError=""
				themeType="dark"
				onSignIn={mockOnSocialSignIn}
				onCancel={mockOnCancel}
				isInternalUser
			/>
		);

		expect(
			screen.getByText(/We will take you to your default web browser/)
		).toBeInTheDocument();
	});

	it("switches to enterprise step when enterprise option is clicked", () => {
		render(
			<SignInSteps
				signInStatus="not-started"
				signInError=""
				themeType="dark"
				onSignIn={mockOnSocialSignIn}
				onCancel={mockOnCancel}
				isInternalUser
			/>
		);

		fireEvent.click(screen.getByText(/Sign in with AWS IAM Identity Center/));

		expect(screen.getByLabelText(/Start URL/)).toBeInTheDocument();
		expect(screen.getByLabelText(/Region/)).toBeInTheDocument();
	});

	it("calls onSocialSignIn when social provider is selected", () => {
		render(
			<SignInSteps
				signInStatus="not-started"
				signInError=""
				themeType="dark"
				onSignIn={mockOnSocialSignIn}
				onCancel={mockOnCancel}
				isInternalUser
			/>
		);

		const googleButton = screen.getByRole("button", { name: /Google/ });
		fireEvent.click(googleButton);

		expect(mockOnSocialSignIn).toHaveBeenCalledWith("Google");
	});

	it("Can cancel current execution", () => {
		render(
			<SignInSteps
				signInStatus="pending"
				signInError=""
				themeType="dark"
				onSignIn={mockOnSocialSignIn}
				onCancel={mockOnCancel}
				isInternalUser
			/>
		);

		const cancelButton = screen.getByRole("button", { name: /Cancel/ });
		fireEvent.click(cancelButton);

		expect(screen.getByText(/Pro license\?/)).toBeInTheDocument();
	});

	describe("Enterprise Step", () => {
		beforeEach(() => {
			render(
				<SignInSteps
					signInStatus="not-started"
					signInError=""
					themeType="dark"
					onSignIn={mockOnSocialSignIn}
					onCancel={mockOnCancel}
					isInternalUser
				/>
			);

			fireEvent.click(screen.getByText(/Sign in with AWS IAM Identity Center/));
		});

		it("disables submit button when URL is empty", () => {
			const submitButton = screen.getByRole("button", { name: /Continue/ });
			expect(submitButton).toBeDisabled();
		});

		it("enables submit button when URL is provided", () => {
			const urlInput = screen.getByLabelText(/Start URL/);
			fireEvent.change(urlInput, {
				target: { value: "test.example.com/start" },
			});

			const submitButton = screen.getByRole("button", { name: /Continue/ });
			expect(submitButton).not.toBeDisabled();
		});

		it("calls onEnterpriseSignIn with form values when submitted", () => {
			const urlInput = screen.getByLabelText(/Start URL/);
			fireEvent.change(urlInput, {
				target: { value: "test.example.com/start" },
			});

			const submitButton = screen.getByRole("button", { name: /Continue/ });
			fireEvent.click(submitButton);

			expect(mockOnEnterpriseSignIn).toHaveBeenCalledWith({
				url: "test.example.com/start",
				region: "us-east-1",
			});
		});

		it("returns to social login step when back button is clicked", () => {
			const backButton = screen.getByRole("button", {
				name: /View all login options/,
			});
			fireEvent.click(backButton);

			expect(screen.getByText(/Pro license\?/)).toBeInTheDocument();
		});
	});
});
