import { ReactElement, useState } from "react";
import { KiroButton } from '../kiro-button';

interface SignInInputProps {
	id: string;
	name: string;
	label: string;
	submitButtonText: string;
	helperText?: string;
	errorText?: string;
	onSubmit: (value: string) => void;
	disabled?: boolean;
	placeholder?: string;
	autoFocus?: boolean;
	loading?: boolean;
}

/**
 * Text input for Sign in experience
 */
export function SignInSingleInputForm({
	id,
	name,
	label,
	submitButtonText,
	helperText,
	errorText,
	onSubmit,
	disabled,
	placeholder,
	autoFocus,
	loading
}: SignInInputProps): ReactElement {
	const [value, setValue] = useState<string>("");

	return (
		<form
			className="w-[420px]"
			onSubmit={() => {
				onSubmit(value);
			}}
		>
			<label htmlFor={id} className="text-[14px]/5 font-medium">
				{label}
			</label>
			{helperText && (
				<p
					id={`${id}_helperText`}
					className="text-[12px]/5 font-normal secondary-text"
				>
					{helperText}
				</p>
			)}
			<div className="w-full flex flex-row gap-3 items-stretch mt-2 mb-1">
				<input
					id={id}
					aria-describedby={`${id}_helperText`}
					type="text"
					value={value}
					name={name}
					disabled={disabled}
					autoFocus={autoFocus}
					placeholder={placeholder}
					className={`grow rounded-lg sign-in-input p-3 w-full text-[14px]/6 font-normal disabled:text-kiro-sign-in-social-input-disabled-text disabled:bg-kiro-sign-in-social-input-disabled-background${errorText ? " error" : ""
						}`}
					onChange={(event) => {
						setValue(event.target.value);
					}}
				></input>
				<div className="w-[200px] flex items-stretch">
					<KiroButton disabled={!value} type="submit" loading={loading}>
						<div className="flex items-center justify-center gap-1">
							<span>{submitButtonText}</span><i className="codicon codicon-arrow-right"></i>
						</div>
					</KiroButton>
				</div>
			</div>

			{errorText && <p className="error-text flex items-center gap-1"><i className="codicon codicon-error"></i><span>{errorText}</span></p>}
		</form>
	);
}
