import { ReactElement } from "react";
import ghost1DarkSrc from "../kiro-logo/ghost1_dark.svg";
import ghost1LightSrc from "../kiro-logo/ghost1_light.svg";
import {
	SignInOptions,
	SignInStatus,
	SocialSignInOptions,
} from "../../types/sign-in-types";
import { SignInSingleInputForm } from "../sign-in-single-input-form";
import { ThemeType } from "../../types/shared-types";

const WAITLIST_URL = "https://kiro.dev/waitlist";
const ACCESS_CODE_PLACEHOLDER = "KIRO-XXXX-XXXX";

interface SignInInvitationProps {
	signInOptions: SignInOptions;
	signInStatus: SignInStatus;
	errorText?: string;
	themeType: ThemeType;
	onSignIn: (signInOptions: SignInOptions) => void;
	onCancel: () => void;
}

export function SignInInvitation({
	signInOptions,
	signInStatus,
	errorText,
	themeType,
	onSignIn,
	onCancel,
}: SignInInvitationProps): ReactElement {
	const isLoading = signInStatus === "pending";
	const isDisabled = isLoading || signInStatus === "success";

	return (
		<div className="text-left">
			<button
				onClick={onCancel}
				disabled={isDisabled}
				aria-label="Go back"
				className={`cursor-pointer text-kiro-button-link mb-3 ${
					isDisabled ? "invisible" : ""
				}`}
			>
				{
					// Using !important here because codicon comes with a default font-size of 16px but its CSS definition is using
					// a higher specificity than what tailwind is usually producing.
				}
				<i className="codicon codicon-arrow-left !text-[24px]/10"></i>
			</button>

			<main className="flex items-center flex-col gap-4" role="main">
				<img
					alt="Kiro"
					src={
						themeType === "dark" || themeType === "hcDark"
							? ghost1DarkSrc
							: ghost1LightSrc
					}
				/>
				<h2 className="link-button text-[30px]/9 font-normal text-wrap text-center">
					Looks like you're new here...
				</h2>
				<p className="w-full max-w-[500px] px-4 mb-5 secondary-text text-[16px]/6 font-normal text-wrap text-center">
					Kiro currently requires an access code to use.
					<br />
					If you've received a code via email, enter it below.
				</p>
				<SignInSingleInputForm
					id="kiro-access-code-input"
					label="Kiro Access Code"
					name="kiro-access-code"
					errorText={errorText}
					placeholder={ACCESS_CODE_PLACEHOLDER}
					onSubmit={(value) => {
						onSignIn({
							...(signInOptions as SocialSignInOptions),
							invitationCode: value,
						});
					}}
					loading={isLoading}
					submitButtonText="Validate"
					autoFocus
				/>
				<p className="w-full max-w-[500px] px-4 secondary-text text-[16px]/6 font-normal text-wrap text-center mt-3">
					If you don't have a code please{" "}
					<a
						href={WAITLIST_URL}
						target="_blank"
						rel="noopener noreferrer"
						className="underline hover:no-underline link-button"
					>
						join our waitlist
					</a>
					.
				</p>
			</main>
		</div>
	);
}
