import { ReactElement } from "react";
import googleImgSrc from "./web_dark_rd_na.svg";
import githubDarkImgSrc from "./github-dark.svg";
import githubLightImgSrc from "./github-light.svg";
import awsDarkImgSrc from "./aws-dark.svg";
import awsLightImgSrc from "./aws-light.svg";
import { SignInOptions } from "../../types/sign-in-types";
import { ThemeType } from "../../types/shared-types";

type Provider = SignInOptions["provider"];

interface SignInButtonProps {
	onClick: () => void;
	themeType: ThemeType;
	provider: Provider;
}

interface ProviderConfig {
	label: string;
	iconPath?: {
		light: string;
		dark: string;
	};
}

const providerConfigMap: Record<Provider, ProviderConfig> = {
	Google: {
		label: "Sign in with Google",
		iconPath: {
			light: googleImgSrc,
			dark: googleImgSrc,
		},
	},
	Github: {
		label: "Sign in with GitHub",
		iconPath: {
			light: githubLightImgSrc,
			dark: githubDarkImgSrc,
		},
	},
	BuilderId: {
		label: "Sign in with AWS Builder ID",
		iconPath: {
			light: awsLightImgSrc,
			dark: awsDarkImgSrc,
		},
	},
	Internal: {
		label: "Sign in with Midway",
	},
	Enterprise: {
		label: "Sign in with your organization identity",
	},
};

/**
 * Button used for social sign-in options
 */
export function SignInButton({
	provider,
	themeType,
	onClick,
}: SignInButtonProps): ReactElement {
	const label = providerConfigMap[provider].label || `Sign in with ${provider}`;
	const category =
		themeType === "dark" || themeType === "hcDark" ? "dark" : "light";

	return (
		<button
			className="gap-3 rounded-lg sign-in-button flex justify-center items-center p-2 w-full text-[14px]/6 font-medium cursor-pointer"
			onClick={onClick}
		>
			{providerConfigMap[provider].iconPath ? (
				<img src={providerConfigMap[provider].iconPath[category]} width="20" />
			) : null}
			<span>{label}</span>
		</button>
	);
}
