import { ReactElement } from "react";
import { ColorTheme } from "../../types/shared-types";
import { ThemePreview } from "./theme-preview";

interface ThemeCardProps {
	theme: ColorTheme;
	selected?: boolean;
	loading?: boolean;
	onSelect: () => void;
}

function ThemeCard({
	theme,
	selected,
	loading,
	onSelect,
}: ThemeCardProps): ReactElement {
	return (
		<button
			disabled={loading}
			onClick={onSelect}
			className={`p-4 rounded-3xl glass-card clickable flex flex-col gap-2 border-1 border-solid ${
				selected ? "border-blue-300" : "cursor-pointer border-transparent"
			}`}
		>
			<div className={`relative text-[14px]/5 font-normal`}>
				<ThemePreview theme={theme} loading={loading} />
			</div>
			<h2 className="text-[14px]/5 font-normal">{theme.label}</h2>
		</button>
	);
}

interface ThemePickerProps {
	importedTheme?: string;
	currentThemeId: string;
	themes: ColorTheme[];
	onChange: (theme: string) => void;
}

export function ThemePicker({
	importedTheme,
	currentThemeId,
	themes,
	onChange,
}: ThemePickerProps): ReactElement {
	return (
		<ul className="list-none flex gap-3 justify-center">
			{themes.map((theme) => (
				<li key={theme.id}>
					<ThemeCard
						theme={theme}
						selected={currentThemeId === theme.id}
						onSelect={() => onChange(theme.id)}
					/>
				</li>
			))}
			{importedTheme &&
				!themes.some((theme) => theme.label === importedTheme) && (
					<li>
						<ThemeCard
							theme={{
								id: "__TEMP__",
								label: importedTheme,
								type: "dark",
								getColor() {
									return "transparent";
								},
								getTokenCSS() {
									return "";
								},
							}}
							onSelect={() => {}}
							loading
						/>
					</li>
				)}
		</ul>
	);
}
