import { ReactElement, useContext, useEffect, useState } from "react";
import { OnboardingNavigation } from "../onboarding-navigation";
import { OnboardingStepSettings } from "./onboarding-step-settings";
import { OnboardingStepTheme } from "./onboarding-step-theme";
import { OnboardingStepCli } from "./onboarding-step-cli";

import iconSettingsLogoPath from "./icon-settings.svg";
import iconCliLogoPath from "./icon-cli.svg";
import iconThemeLogoPath from "./icon-theme.svg";

import iconSettingsLightLogoPath from "./icon-settings-light.svg";
import iconCliLightLogoPath from "./icon-cli-light.svg";
import iconThemeLightLogoPath from "./icon-theme-light.svg";

import { OnboardingStepLogo } from "./onboarding-step-logo";
import { ColorTheme, ThemeType } from "../../types/shared-types";

interface OnboardingStepsProps {
	onCompleted: () => void;
	themes: ColorTheme[];
	currentThemeId: string;
	themeType: ThemeType;
}

interface OnboardingStep {
	icon: string;
	render: () => React.ReactNode;
}

export function OnboardingSteps({
	onCompleted,
	themes,
	currentThemeId,
	themeType,
}: OnboardingStepsProps): ReactElement {
	const [stepIndex, setStepIndex] = useState<number>(0);
	const [visitedCount, setVisitedCount] = useState<number[]>([1, 0, 0]);

	const isDark = themeType === "dark" || themeType === "hcDark";

	async function complete() {
		onCompleted();
	}

	function goToStep(index: number): void {
		setVisitedCount((oldCount) => {
			const oldVisitedCount = oldCount[index] || 0;
			const newCount = [...oldCount];
			newCount[index] = oldVisitedCount + 1;
			return newCount;
		});
		setStepIndex(index);
	}

	function isFirstTimeVisit(): boolean {
		return (visitedCount[stepIndex] || 0) <= 1;
	}

	function autoProgress(nextStepIndex: number): void {
		// We are only automatically progressing to the next step if the user is still on the current step and hasn't
		// moved forward manually AND if the user visited to current step only a single time.
		// Reason: if they visited multiple times, it means that they intentionally went back to a previous step,
		//         so we keep them there until they make another intentionaly choice.
		if (nextStepIndex === stepIndex + 1 && isFirstTimeVisit()) {
			// Leave some time for animation to complete that shows the step as complete
			setTimeout(() => {
				nextStepIndex >= steps.length ? complete() : goToStep(nextStepIndex);
			}, 750);
		}
	}

	const steps: OnboardingStep[] = [
		{
			icon: isDark ? iconSettingsLogoPath : iconSettingsLightLogoPath,
			render: () => (
				<OnboardingStepSettings
					onComplete={() => autoProgress(1)}
					onNext={() => goToStep(1)}
				/>
			),
		},
		{
			icon: isDark ? iconThemeLogoPath : iconThemeLightLogoPath,
			render: () => (
				<OnboardingStepTheme
					themes={themes}
					currentThemeId={currentThemeId}
					onNext={() => goToStep(2)}
				/>
			),
		},
		{
			icon: isDark ? iconCliLogoPath : iconCliLightLogoPath,
			render: () => (
				<OnboardingStepCli
					onComplete={() => autoProgress(3)}
					onNext={complete}
				/>
			),
		},
	];

	const step = steps[stepIndex];

	return (
		<div className="h-full w-full relative flex flex-col items-center">
			<main
				className={`w-screen grow flex items-center justify-center flex-col gap-8`}
			>
				<OnboardingStepLogo icon={step.icon} />
				{step.render()}
			</main>
			<div className="mb-20">
				<OnboardingNavigation
					numberOfSteps={steps.length}
					currentStep={stepIndex}
					onStepChange={goToStep}
					onSkipAll={complete}
				/>
			</div>
		</div>
	);
}
