import { ReactElement } from "react";
import { LinkButton } from "../link-button";

interface NavigationButtonProps {
	selected: boolean;
	onClick: () => void;
}

function NavigationButton({
	selected,
	onClick,
}: NavigationButtonProps): ReactElement {
	return (
		<button
			disabled={selected}
			onClick={onClick}
			className={`w-4 h-4 border-none relative ${
				!selected ? "cursor-pointer" : ""
			}`}
		>
			<span
				className={`block absolute left-1 top-1 w-2 h-2 rounded-sm border-none ${
					selected ? "primary-button" : "bg-[#9a9a9a]"
				}`}
			></span>
		</button>
	);
}

interface OnboardingNaviationProps {
	currentStep: number;
	numberOfSteps: number;
	onStepChange: (index: number) => void;
	onSkipAll: () => void;
}

export function OnboardingNavigation({
	currentStep,
	numberOfSteps,
	onStepChange,
	onSkipAll,
}: OnboardingNaviationProps): ReactElement {
	return (
		<nav className="flex gap-5 items-center text-[12px]">
			<div className="w-16 text-right inline">
				{currentStep > 0 && (
					<button
						onClick={() => onStepChange(currentStep - 1)}
						className="inline link-button bg-inherit border-none cursor-pointer"
					>
						<span className="flex items-center gap-1 items-center">
							<i className="codicon codicon-arrow-left"></i>
							<span>Back</span>
						</span>
					</button>
				)}
			</div>

			<ul className="flex gap-5 list-none">
				{new Array(numberOfSteps).fill(0).map((_, index) => (
					<li key={index} className="flex items-center">
						<NavigationButton
							selected={index === currentStep}
							onClick={() => onStepChange(index)}
						></NavigationButton>
					</li>
				))}
			</ul>

			<div className="w-16 text-left inline">
				<button
					onClick={() => onSkipAll()}
					className="inline link-button bg-inherit border-none cursor-pointer"
				>
					<span className="flex items-center gap-1 items-center">
						<span>Skip All</span>
						<i className="codicon codicon-arrow-right"></i>
					</span>
				</button>
			</div>
		</nav>
	);
}
