import { ReactElement, useContext, useState } from "react";
import { KiroGhostBlinking } from "../kiro-ghost-blinking";
import type { UserInfo } from "../../types/account-dashboard-types";
import { AccountDashboardApiContext } from "../../context/account-dashboard-api-context";
import { logError } from "../../utils/logger";
import { UserIdCopy } from "./user-id-copy";

interface UserInfoProps {
	userInfo?: UserInfo;
}

export function UserInfo({ userInfo }: UserInfoProps): ReactElement {
	const { close, signOut } = useContext(AccountDashboardApiContext);

	const [signingOut, setSigningOut] = useState(false);

	const handleSignOutClick = async () => {
		if (!signOut || signingOut) return;

		try {
			setSigningOut(true);
			await signOut();
			close();
		} catch (error) {
			logError("Sign out failed", error as Error);
		} finally {
			setSigningOut(false);
		}
	};

	return (
		<div className="kiro-flex-between">
			<KiroGhostBlinking withPurpleBackground={true} size="sm" />
			{userInfo && (
				<div className="kiro-flex-col">
					<span className="kiro-text-primary" data-testid="user-email">
						{userInfo.email || "Kiro"}
					</span>
					<div className="kiro-flex kiro-items-center kiro-gap-xs">
						{userInfo.provider && (
							<span className="kiro-text-secondary" data-testid="user-provider">
								Signed in with {userInfo.provider}
							</span>
						)}
						{userInfo.userId && (
							<>
								{userInfo.provider && (
									<span className="kiro-text-secondary kiro-divider">|</span>
								)}
								<UserIdCopy userId={userInfo.userId} />
							</>
						)}
					</div>
				</div>
			)}
			<button
				onClick={handleSignOutClick}
				disabled={signingOut || !signOut}
				data-testid="sign-out-button"
				className="kiro-button-outline"
			>
				{signingOut ? "Signing Out..." : "Sign Out"}
			</button>
		</div>
	);
}
