import { ReactElement } from "react";
import { UsageLimits } from "../../types/account-dashboard-types";
import {
	formatResetDate,
	isUsageItemNotAvailable,
	filterAvailableUsageBreakdowns,
	shouldShowWelcomeBonus,
} from "../../utils/subscription-utils";
import { UsageBreakdownView } from "./components/usage-breakdown";
import { WelcomeBonus } from "./components/welcome-bonus";

interface UsageSummaryProps {
	usageLimits: UsageLimits | null;
}

export function UsageSummary({ usageLimits }: UsageSummaryProps): ReactElement {
	if (!usageLimits) {
		return (
			<div className="kiro-usage-summary-container kiro-usage-summary-no-data">
				<span className="kiro-text-secondary">
					No usage information available right now.
				</span>
			</div>
		);
	}

	const availableBreakdowns = filterAvailableUsageBreakdowns(
		usageLimits.usageBreakdowns
	);

	const showWelcomeBonus = shouldShowWelcomeBonus(usageLimits.usageBreakdowns);

	return (
		<div>
			{showWelcomeBonus && (
				<WelcomeBonus trialBreakdowns={usageLimits.usageBreakdowns} />
			)}

			<div className="kiro-usage-summary-header">
				<div>
					<div className="kiro-usage-summary-title-section">
						<h3 className="kiro-text-primary">Estimated Usage</h3>
						{usageLimits.billingCycle?.resetDate && (
							<span className="kiro-text-secondary">
								resets on{" "}
								{formatResetDate(new Date(usageLimits.billingCycle.resetDate))}
							</span>
						)}
					</div>
				</div>
				<div className="kiro-usage-summary-subscription-section">
					<div className="kiro-usage-summary-subscription-row">
						<span
							className={`kiro-subscription-badge ${
								// Using overageCapable since there is no better field and this should be true only for non-free tier
								!usageLimits.subscriptionInfo.overageCapable
									? "kiro-subscription-badge--free"
									: "kiro-subscription-badge--premium"
							}`}
						>
							{usageLimits.subscriptionInfo.subscriptionName}
						</span>
					</div>
				</div>
			</div>

			<div className="kiro-usage-summary-content">
				{!availableBreakdowns.length ? (
					<div className="kiro-usage-summary-empty-state">
						<span className="kiro-text-secondary">No usage data available</span>
					</div>
				) : (
					<div className="kiro-usage-summary-content-spacing">
						{availableBreakdowns.map((item, index) => (
							<UsageBreakdownView
								key={`usage-${index}`}
								item={{
									...item,
									currentOverages: item.currentOverages || 0,
									overageCap: item.overageCap || item.usageLimit,
								}}
								index={index}
								isDisabled={isUsageItemNotAvailable(item)}
							/>
						))}
					</div>
				)}
			</div>
		</div>
	);
}
