import { ReactElement, useContext } from "react";
import { AccountDashboardApiContext } from "../../context/account-dashboard-api-context";
import { UsageLimits } from "../../types/account-dashboard-types";
import { logError } from "../../utils/logger";

interface PlanActionsProps {
	usageLimits: UsageLimits | null;
	onClose: () => void;
}

export function PlanActions({
	usageLimits,
	onClose,
}: PlanActionsProps): ReactElement {
	const { showSubscriptionPlan, getPaymentPortalUrl, openExternalLink } =
		useContext(AccountDashboardApiContext);

	const shouldShowUpgradeplanButton =
		usageLimits?.subscriptionInfo.subscriptionManagementTarget === "PURCHASE";

	const handleShowPlansModal = async () => {
		try {
			showSubscriptionPlan();
		} catch (error) {
			logError("Failed to show subscription plans", error as Error);
		}
	};

	const handleManagePlanPortal = async () => {
		try {
			const response = await getPaymentPortalUrl();
			if (response.error) {
				logError(`Failed to get payment portal URL: ${response.error}`);
				return;
			}
			if (response.data) {
				await openExternalLink(response.data);
				onClose();
			}
		} catch (error) {
			logError("Failed to manage plan portal", error as Error);
		}
	};

	const handleContactSupport = async () => {
		try {
			await openExternalLink("https://support.aws.amazon.com/#/contacts/kiro");
		} catch (error) {
			logError("Failed to open billing support link", error as Error);
		}
	};

	// Reusable plan button component
	const PlanButton = ({
		title,
		onClick,
	}: {
		title: string;
		onClick: () => void;
	}) => (
		<button onClick={onClick} className="kiro-plan-button">
			{title}
		</button>
	);

	return (
		<div className="kiro-plan-actions-container">
			{shouldShowUpgradeplanButton ? (
				<PlanButton title={"Upgrade Plan"} onClick={handleShowPlansModal} />
			) : (
				<PlanButton title={"Manage Plan"} onClick={handleManagePlanPortal} />
			)}

			<span onClick={handleContactSupport} className="kiro-support-link">
				Contact Billing Support
			</span>
		</div>
	);
}
