import { ReactElement } from "react";
import { useOverageSettings } from "../../hooks/useOverageSettings";
import { formatResourceType } from "../../utils/subscription-utils";
import { UsageLimits } from "../../types/account-dashboard-types";

interface OverageSettingsProps {
	usageLimits: UsageLimits | null;
}

export function OverageSettings({
	usageLimits,
}: OverageSettingsProps): ReactElement | null {
	const {
		overageEnabled,
		updating,
		handleOverageToggle,
		resourceToOverageMap,
		shouldShowOverageTable,
		formatCurrency,
		generateOverageText,
	} = useOverageSettings({
		usageLimits,
	});

	const overageData = Array.from(resourceToOverageMap);
	const [firstResourceType, firstDatum] = overageData[0] ?? [];

	return (
		<div className="kiro-overage-settings">
			<div className="kiro-overage-settings-header">
				<span className="kiro-text-primary">Overages</span>
				<label
					className={`kiro-overage-toggle-wrapper ${
						updating ? "kiro-overage-toggle-wrapper--disabled" : ""
					}`}
				>
					<input
						type="checkbox"
						checked={overageEnabled}
						onChange={(e) => handleOverageToggle(e.target.checked)}
						disabled={updating}
						className="kiro-overage-toggle-input"
					/>
					<div
						className={`kiro-overage-toggle-track ${
							overageEnabled ? "kiro-overage-toggle-track--enabled" : ""
						} ${updating ? "kiro-overage-toggle-track--disabled" : ""}`}
					>
						<div
							className={`kiro-overage-toggle-thumb ${
								overageEnabled ? "kiro-overage-toggle-thumb--enabled" : ""
							}`}
						></div>
					</div>
				</label>
			</div>

			<div className="kiro-text-secondary">
				Turn on overages to keep using Kiro past these limits
				<br />({generateOverageText()}).
			</div>

			{/* Overage Usage Table - Only show if there are overages */}
			{shouldShowOverageTable && (
				<div className="kiro-overage-table-wrapper">
					<table className="kiro-overage-table">
						{overageData.length > 1 ? (
							<>
								<thead>
									<tr className="kiro-overage-table-header">
										<th className="kiro-overage-table-header-left" />
										<th className="kiro-overage-table-header-right">
											Est. Overages
										</th>
										<th className="kiro-overage-table-header-right">
											Est. Overage Billing
										</th>
									</tr>
								</thead>
								<tbody>
									{overageData.map(([resourceType, data]) => {
										const isTotal = resourceType === "TOTAL";

										return (
											<tr
												key={`${resourceType}-overages`}
												className={`kiro-overage-table-row ${
													isTotal ? "kiro-overage-table-row--total" : ""
												}`}
											>
												<td
													className={`kiro-overage-table-cell-left ${
														isTotal ? "kiro-overage-table-cell--total" : ""
													}`}
												>
													{data.displayNamePlural ||
														formatResourceType(resourceType)}
												</td>
												<td
													className={`kiro-overage-table-cell-right ${
														isTotal ? "kiro-overage-table-cell--total" : ""
													}`}
												>
													{isTotal ? "" : data.overages}
												</td>
												<td
													className={`kiro-overage-table-cell-right ${
														isTotal ? "kiro-overage-table-cell--total" : ""
													}`}
												>
													{formatCurrency(data.overageBilling)}
												</td>
											</tr>
										);
									})}
								</tbody>
							</>
						) : (
							<tbody>
								<tr
									key={`${firstResourceType}-overages`}
									className="kiro-overage-table-row--single-resource"
								>
									<td className="kiro-overage-table-cell-left">
										{`Est. ${
											firstDatum.displayNamePlural ||
											formatResourceType(firstResourceType)
										} Used`}
									</td>
									<td className="kiro-overage-table-cell-right">
										{firstDatum.overages}
									</td>
								</tr>
								<tr
									key={`${firstResourceType}-overage-billing`}
									className="kiro-overage-table-row--single-resource"
								>
									<td className="kiro-overage-table-cell-left">
										Est. Overage Billing
									</td>
									<td className="kiro-overage-table-cell-right">
										{formatCurrency(firstDatum.overageBilling)}
									</td>
								</tr>
							</tbody>
						)}
					</table>
				</div>
			)}
		</div>
	);
}
