import { ReactElement, useContext } from "react";
import { UsageSummary } from "./usage-summary";
import { OverageSettings } from "./overage-settings";
import { PlanActions } from "./plan-actions";
import { SignInPrompt } from "./sign-in-prompt";
import { SectionContainer, PageWrapper } from "./components";
import { LoadingSkeleton } from "./components/loading-skeleton";
import { AccountDashboardApiContext } from "../../context/account-dashboard-api-context";
import { useUsageLimits } from "../../hooks/useUsageLimits";

export function AccountDashboardPage(): ReactElement {
	const { close } = useContext(AccountDashboardApiContext);
	const { usageLimits, loading, error } = useUsageLimits();

	const userInfo = usageLimits?.userInfo;
	const isLoggedIn = userInfo?.isLoggedIn ?? false;

	// Show error state
	if (error) {
		// For 403 errors with a reason, show only PageWrapper without error message
		if (
			error.statusCode &&
			error.statusCode === 403 &&
			error.reason &&
			error.reason === "FEATURE_NOT_SUPPORTED"
		) {
			const errorUserInfo = {
				email: "",
				isLoggedIn: error.isLoggedIn ?? false,
				provider: error.provider,
			};
			return <PageWrapper userInfo={errorUserInfo}>{null}</PageWrapper>;
		}

		return (
			<PageWrapper userInfo={userInfo}>
				<SectionContainer showSeparatorBefore>
					<div className="text-[var(--vscode-errorForeground)] text-center">
						{error.message}
					</div>
				</SectionContainer>
			</PageWrapper>
		);
	}

	// Show loading state
	if (loading) {
		return <LoadingSkeleton />;
	}

	// Show sign-in prompt if user is not logged in
	if (!isLoggedIn) {
		return (
			<SectionContainer>
				<SignInPrompt />
			</SectionContainer>
		);
	}

	// Show only UserInfo for Internal sign-in provider
	if (userInfo?.signInProvider === "Internal") {
		return <PageWrapper userInfo={userInfo} children={undefined} />;
	}

	// Show limited dashboard for standalone subscription (not available)
	if (
		usageLimits?.subscriptionInfo.subscriptionType === "Q_DEVELOPER_STANDALONE"
	) {
		return (
			<PageWrapper userInfo={userInfo}>
				<SectionContainer showSeparatorBefore>
					<div className="rounded-lg border border-[var(--vscode-menu-border)] p-6">
						<div className="text-center py-8">
							<div className="text-[var(--vscode-descriptionForeground)] mb-4">
								<i
									className="codicon codicon-warning"
									style={{ fontSize: "32px" }}
								></i>
							</div>
							<h3 className="text-lg font-medium mb-2">
								Subscription Plan Not Available
							</h3>
							<span className="text-sm text-[var(--vscode-descriptionForeground)]">
								Your current subscription plan is no longer available. Please
								contact support for assistance with your account.
							</span>
						</div>
					</div>
				</SectionContainer>
			</PageWrapper>
		);
	}

	// Show full dashboard for logged-in users
	return (
		<PageWrapper userInfo={userInfo}>
			<SectionContainer showSeparatorBefore>
				<UsageSummary usageLimits={usageLimits} />
			</SectionContainer>

			{/* Don't show overage settings for free tier users */}
			{usageLimits && usageLimits.subscriptionInfo.overageCapable && (
				<SectionContainer showSeparatorBefore>
					<OverageSettings usageLimits={usageLimits} />
				</SectionContainer>
			)}

			<PlanActions usageLimits={usageLimits} onClose={close} />
		</PageWrapper>
	);
}
